/***************************************************************
 *                 Mathematical Object Library                 *
 *        Combinatorial Analysis Library - Include File        *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004,2005,2006,2007,2008,2009 COLLARD Christophe
 * copyright © 2004,2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::combinatorial_analysis
    \brief combinatorial analysis library \n

    \htmlonly 
    <FONT color="#838383">

    combinatorial_analysis belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006, 2007, 2008, 2009 Christophe COLLARD \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.4.0
    \date 2004-2010
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type combination
#endif

#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__PARAMETERS_H)
#include "../parameters.h"
#endif

#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "matrix.h"
#endif

#ifndef _combination_h
#define _combination_h

using namespace std;

namespace mol
{


//=============================================
template <class T> class combinatorial_analysis
//=============================================
{
  private :
    void combinations (int, const vector<T>&, vector<T>&, matrix<T>&, int&,  int=1);

  public :
    friend unsigned long int Cnp (int, int);
    template <class M> friend matrix<M> combinations (int, const vector<M>&);
};


//=====Private methods for combinatorial analysis===========================================


//----------------------------------------------------------------------------------------------
template <class T> void combinatorial_analysis<T>::combinations
(int n, const vector<T>& systems, vector<T>& x, matrix<T>& combinations_matrix, int& k, int dim)
//----------------------------------------------------------------------------------------------
{
  for (int i=dim; i<=systems.dim(); i++)
    { x[n] = systems[i];
      if (n>1) combinations (n-1,systems,x,combinations_matrix,k,i+1);
      else combinations_matrix[k++] = x;
    }
}


//=====Public methods for combinatorial analysis============================================

/*!
  \brief Computes \f$ \displaystyle C_n^p = \frac{n!}{p!(n-p!)} \f$
  \param n total number of objects
  \param p number of elements in a group of objects
  \return q : number of arrangements of two groups of p and n-p indistinguishable objects
*/

//----------------------------------
unsigned long int Cnp (int n, int p)
//----------------------------------
{
  assert (p>=0 && n>=p);

  long double q=1;

  for (int i=p+1; i<=n; i++)
    q *= i;

  for (int i=2; i<=(n-p); i++)
    q /= i;

  q = round(q);
  assert (q == (unsigned long int) q);

  return q;
}


/*!
  \brief Computes the arrangements of indistinguishable objects
  \param n number of elements in a group of objects
  \param systems objects
  \return combination_matrix : arrangements of two groups of p and n-p indistinguishable objects
*/

//-------------------------------------------------------------------------
template <class T> matrix<T> combinations (int n, const vector<T>& systems)
//-------------------------------------------------------------------------
{
  assert (n>0 && n<=systems.dim());

  int size = Cnp(systems.dim(),n); // computes C_n^p
  matrix<T> combinations_matrix(size,n,false);
  vector<T> x(n,false);
  int k=1;

  combinatorial_analysis<T> ca;
  ca.combinations (n,systems,x,combinations_matrix,k);

  return combinations_matrix;
}


}


#endif
